///////////////////////////////
//  Makumba, Makumba tag library
//  Copyright (C) 2000-2003  http://www.makumba.org
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
//  -------------
//  $Id: JspPageData.java,v 1.2 2003/02/27 15:22:37 stefan Exp $
//  $Name: makumba-0_5_5_27 $
/////////////////////////////////////

package org.makumba.view.jsptaglib2;
import org.makumba.util.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;

/** This class performs a makumba analysis of a JSP page for determining its correctness, caching of runtime resources such as OQL queries and form responders as well as for syntax colouring. 

 * Every JSP page gets analyzed when its analysis is missing from this Java classloader, or when the page was modified after its last analysis. Every makumba tag invokes JspPageData.getPageData(). Thus the analysis is triggered by the first (prototyped) makumba tag in a page. 

 * @author cristi
 */
public class JspPageData
{
  /** the prefix used for makumba tags in this page */
  String makumbaPrefix;
  
  /** the path of the JSP page */
  String filePath;
  
  /** the timestamp of the JSP page. if the page is found newer on disk, 
    the object is discarded */
  long lastChanged; 

  /** the patterns used to parse the page */
  static final String attribute= "\\s+\\w+\\s*=\\s*\\\"([^\\\"]|\\\\\")*\\\"";
  static final Pattern JspSystemTagPattern= Pattern.compile("<%@\\s*\\w+("+attribute+")*\\s*%>");
  static final Pattern JspTagPattern= Pattern.compile("<((\\s*\\w+:\\w+("+attribute+")*\\s*)/?|(/\\w+:\\w+\\s*))>");
  static final Pattern JspCommentPattern= Pattern.compile("<%--.*--%>", Pattern.DOTALL);

  static final Pattern JspTagAttributePattern= Pattern.compile(attribute);

  static final Pattern Word= Pattern.compile("\\w+");
  static final Pattern TagName= Pattern.compile("\\w+:\\w+");

  /** private  constructor, construction can only be made by getPageData() */
  private JspPageData(String path)
  {
    System.out.println(path);

    filePath=path;
    lastChanged=new File(path).lastModified();

    String content=readFile();    
    
    // add the text lines as syntax points (beginning and end of each line)
    SyntaxPoint.addLines(content, syntaxPoints);

    // remove JSP comments from the text
    content= SyntaxPoint.unComment(content, JspCommentPattern, "JSPComment", syntaxPoints);

    // the page analysis as such:
    treatMakumbaTags(content, findMakumbaPrefix(content));
  }

  /** identify tag attributes from a tag string and put them in a Map. set the attribute syntax points */
  Map parseAttributes(String s, int origin)
  {
    Map ret= new HashMap(13);
    
    Matcher m= JspTagAttributePattern.matcher(s);
    while(m.find())
      {
	// here we have an attribute="value"
	String attr=s.substring(m.start(), m.end());
	int n=attr.indexOf('=');
	String val=attr.substring(n+1).trim();
	
	// we use a streamtokenizer to do the complicated parsing of "...\"\t ...\n...."
	StreamTokenizer st= new StreamTokenizer(new StringReader(val));
	st.quoteChar('\"');
	try{
	  if(st.nextToken()!=StreamTokenizer.TT_WORD)
	    throw new RuntimeException("quoted string expected, found "+val);
	}catch(java.io.IOException ioe) { throw new RuntimeWrappedException(ioe);}
	String attName= attr.substring(0, n).trim();
	ret.put(attName, st.sval);
	
	// syntax points
	SyntaxPoint.addSyntaxPoints(syntaxPoints, origin, origin+attName.length(), "JSPTagAttributeName", null);
	SyntaxPoint.addSyntaxPoints(syntaxPoints, origin+n, origin+n+1, "JSPTagAttributeEquals", null);
	SyntaxPoint.addSyntaxPoints(syntaxPoints, origin+s.indexOf('\"', n), origin+s.length(), "JSPTagAttributeValue", null);
      }
    return ret;
  }

  /** identify the makumba prefix by parsing system tags */
  String findMakumbaPrefix(String content)
  {
    Matcher m= JspSystemTagPattern.matcher(content);
    while(m.find())
      {
	String tag= content.substring(m.start(), m.end());
	SyntaxPoint.addSyntaxPoints(syntaxPoints, m.start(), m.end(), "JSPSystemTag", null);

	Matcher m1= Word.matcher(tag);
	m1.find();
	SyntaxPoint.addSyntaxPoints(syntaxPoints, m.start()+m1.start(), m.start()+m1.end(), "JSPSystemTagName", null);

	Map attrs= parseAttributes(tag, m.start());

	if(!tag.substring(m1.start(), m1.end()).equals("taglib"))
	  continue;
	if(!attrs.get("uri").equals("http://www.makumba.org/presentation"))
	  continue;
	return (String)attrs.get("prefix");
      }
    return null;
  }
  
  // keep all the makumba tags in the page
  List makumbaTags= new ArrayList();

  /** look for makumba tags by parsing xxx:yyy tags, 
   * for cacheable tags, make tag prototypes, 
   * ask them to compute their cached data (queries, etc) */
  void treatMakumbaTags(String content, String prefix)
  {
    List parents=new ArrayList();
    Matcher m= JspTagPattern.matcher(content);
    while(m.find())
      {
	String tag= content.substring(m.start(), m.end());
	Matcher m1= TagName.matcher(tag);
	m1.find();
	SyntaxPoint.addSyntaxPoints(syntaxPoints, m.start()+m1.start(), m.start()+m1.end(), "JSPTagName", null);

	String tagName= tag.substring(m1.start(), m1.end());
	String type=tag.startsWith("</")?"JspTagEnd":(tag.endsWith("/>")?"JspTagSimple":"JspTagBegin");

	boolean isMakumbaTag=  prefix==null || !tagName.startsWith(prefix);
	MakumbaTag t=null;

	// at this point we know the tag text and the tag name
	if(isMakumbaTag)
	  // if it's a body tag end
	  if(tag.startsWith("</"))
	    {
	      t= (MakumbaTag)parents.get(parents.size()-1);
	      t.endAnalysis(this);
	      parents.remove(parents.size()-1);
	    }
	  else
	    {
	      // the tag begins here, parse its attributes
	      Map attrs= parseAttributes(tag, m.start());
	      // make a tag
	      t= MakumbaTag.makeTag(tagName.substring(tagName.indexOf(":")+1));
	      // ... and tell it to analyze
	      t.analyzeInPage(this, attrs, parents);
	      // if it's a body tag, it's probably the parent of others
	      if(!tag.endsWith("/>"))
		parents.add(t);
	      else
		// else it ends right here
		t.endAnalysis(this);
	    }
	// add syntax points for the begining and end of this tag
	SyntaxPoint.addSyntaxPoints(syntaxPoints, m.start(), m.end(), type, t);
      }
  }
  
  /** the cache of tag prototypes kept by this page data*/
  Map cache= new HashMap();

  /** the syntax points of this page, in the order of their page position */
  SortedSet syntaxPoints= new TreeSet();

  /** return the pageData of the page at the given path in the given webapp 
    this is the only way for clients of this class to obtain instances 
    of JspPageData
   */
  static public JspPageData getPageData(String webappRoot, String path)
  {
    path=webappRoot+path;
    return checkPageCache(path, new File(path).lastModified());
  }

  /** this is synchronized, so two simmultaneous accesses would not result in double prototypes */
  static synchronized JspPageData checkPageCache(String s, long l)
  {
    JspPageData pd= (JspPageData)pageCache.get(s);
    if(pd!=null && pd.lastChanged>= l)
      return pd;
    pd= new JspPageData(s);
    pageCache.put(s, pd);
    return pd;
  }

  /** the cache of JSP pages */
  static Map pageCache= new HashMap();

  /** return the content of the JSP file in a string */
  public String readFile()
  {
    StringBuffer sb=new StringBuffer();
    try{
      BufferedReader rd= new BufferedReader(new FileReader(filePath));
      char[] buffer= new char[2048];
      int n;
      while((n=rd.read(buffer))!=-1)
	sb.append(buffer, 0, n);
    }catch(IOException e) { e.printStackTrace(); }
    return sb.toString();
  }

}


