///////////////////////////////
//  Makumba, Makumba tag library
//  Copyright (C) 2000-2003  http://www.makumba.org
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
//  -------------
//  $Id: MakumbaPrototypedTag.java,v 1.2 2003/02/27 15:22:37 stefan Exp $
//  $Name: makumba-0_5_5_27 $
/////////////////////////////////////

package org.makumba.view.jsptaglib2;
import org.makumba.util.*;
import java.util.*;
import javax.servlet.jsp.*;
import javax.servlet.jsp.tagext.*;
import javax.servlet.http.*;

/** a prototyped (cached) makumba tag. 

 * Not all makumba tags are cached. Notably, RESPONSE, LOGIN, LOGOUT are not cached (for now), as there is no specific data that they need to cache. LIST, OBJECT, VALUE, FORM, INPUT are cached.

 * The cache held by an instance of JspPageData stores a "prototype" of every cacheable makumba tag in the page. The prototype is an object of the same class as the tag handler (so related things are kept in the same class), but it never gets executed. It has a "key" associated, given by the values of its constant attributes (attributes that can't have runtime values e.g. FROM, WHERE, ORDERBY, GROUPBY for a LIST). Whenever a makumba tag handler execution starts (doStartTag()), it computes its key and uses it to find its prototype in the JspPageData of the page being executed. Once it got hold of its prototype, the tag copies from the prototype the references to the cached resoures that it needs for execution.
 * 
 * some methods are called only at page analysis time, some only at page runtime (doStartTag, etc)
 * 
 * Since the key of a tag is composed by the value of its attributes, no two cacheable makumba tags with the same attributes can exist in a page. Such a case will be signaled as error. This limitation is due to the JSP standard not providing any other way to identify tags in a page.

 * If the page analysis fails for makumba-related reasons, the programmer will see a makumba error. The error can be shown in the source, near the respective tag(s), as the errors are stored in association with the tag that produced them. The page analysis can also produce hints for the programmer and store them in the respective tag. The tags can produce runtime information for the programmer and show it in the resulting page.


 */
public abstract class MakumbaPrototypedTag extends MakumbaTag
{
  /** the prototype of this tag, stored in the page data, null if this tag is a prototype itself */
  MakumbaTag prototype;

  /** the tag key. if this tag is not a prototype, it will discover its prototype with the help of this key */
  MultipleKey key; 

  /** the constructor */
  public MakumbaPrototypedTag()
  {
    key= new org.makumba.util.MultipleKey(getConstantAttributeNames().length+3);
    // the first element of the key is the name of the class
    key.setAt(getClass().getName(), 0);
    // the second is the key of the parent tag, null for now
    key.setAt(null, 1);
    // the third is the id, it will be set by the setter method if indicated, see setId()

    // the rest of the key elements are the constant tag attributes. they are set by the setXXX setter methods (setFrom, setWhere, etc), which are supposed to invoke setConstantAttribute for constant attributes
  }

  /** invoked by the JSP engine. complete the key computation (started in the constructor) and find the prototype. then start the execution, which will use data stored in the prototype by the page analysis*/
  public int doStartTag() throws JspException
  {
    // at this point, we should know the parent tag (setParent() is invoked before doStartTag()), so we can add it to the key
    setParentInKey();

    String pageName=((HttpServletRequest)pageContext.getRequest()).getServletPath();
    // now the key is complete, we determine JspPageData, then the prototype
    prototype= (MakumbaTag)JspPageData.getPageData(pageContext.getServletContext().getRealPath("/"), pageName).cache.get(key);
    if(prototype==null)
      {
	treatException( new org.makumba.MakumbaError("prototype not found in page "+pageName+" key "+key));
	return SKIP_PAGE;
      }
    // we have a prototype, we can proceed with execution. this will invoke doStartTag1, which subclasses should define
    return super.doStartTag();
  }

  /** the names of tag attributes that are constant at every page execution. the names depend on the tag type, each tag class will define this method*/
  public abstract String[] getConstantAttributeNames();

  /** during page analysis: once the parent and attributes are known, this method is called to allow the tag to build its cached info */
  public abstract void buildPrototypeInfo();

  /** set the value of a constant attribute. should be used by all setter methods of constant attributes */
  protected void setConstantAttribute(String name, String value)
  {
    setConstantAttribute(getConstantAttributeIndex(name), value);
  }

  /** used by methods in this class */
  void setConstantAttribute(int n, String value)
  {
    key.setAt(value, n);
  }

  /** set the id, assign it to its reserved place in the key */
  public void setId(String id)
  {
    super.setId(id);
    setConstantAttribute(2, id);
  }

  /** retrieve a constant attribute  */
  protected String getConstantAttribute(String name)
  {
    return (String)key.elementAt(getConstantAttributeIndex(name));
  }

  /** every constant attribute is stored in the key at the same position. this method does a search through the array returned by getConstantAttributeNames(). the search is linear since we really have a small number of attributes 
   * -1 indicates runtime (non-constant) atribute
   */
  protected int getConstantAttributeIndex(String attrName)
  {
    if(attrName.equals("id"))
      return 2;
    String keyNames[]= getConstantAttributeNames();
    for(int i=0; i<keyNames.length; i++)
      if(keyNames[i].equals(attrName))
	 return i+3;
    return -1;
  }

  /** find the parent makumba tag, then add the parent key to this tag's key */
  void setParentInKey()
  {
    MakumbaPrototypedTag parent= (MakumbaPrototypedTag)findAncestorWithClass(this, MakumbaPrototypedTag.class);
    if(parent!=null)
      key.setAt(parent.key, 1);    
  }

  /** do analysis during page parsing. */
  public void analyzeInPage(JspPageData pageData, Map attributes, List parents)
  {
    /* go through the attributes discovered by the page analyzer */
    for(Iterator i= attributes.entrySet().iterator(); i.hasNext();)
      {
	Map.Entry me=(Map.Entry)i.next();
	/* test if the attribute is a constant one */
	int n= getConstantAttributeIndex((String)me.getKey());
	if(n==-1)
	  continue;
	/* if constant, set it, so it'll be added to the analysis-time key */
	setConstantAttribute(n, (String)me.getValue());
      }

    /* set the parent */
    setParent((MakumbaTag)parents.get(parents.size()-1));
    setParentInKey();

    /* add this tag (a prototype) to the page cache using the computed key. if a prototype already exists for the same key, we have a problem */
    if(pageData.cache.put(key, this)!=null)
      // a more special error should be thrown
      throw new org.makumba.MakumbaError("two tags with the same key "+key);
    buildPrototypeInfo();
  }
}
