// ========================================================================
// Copyright 2006 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at 
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//========================================================================

package org.mortbay.cometd;

import java.io.IOException;
import java.lang.reflect.Array;
import java.util.Map;
import java.util.Queue;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.mortbay.cometd.filter.JSONDataFilter;
import org.mortbay.util.ajax.JSON;

/**
 * Cometd Filter Servlet implementing the {@link Bayeux} protocol.
 * 
 * The Servlet can be initialized with a json file mapping channels to
 * {@link DataFilter} definitions. The servlet init parameter "filters" should
 * point to a webapplication resource containing a JSON array of filter
 * definitions. For example:
 * 
 * <pre>
 *  [
 *    { 
 *      &quot;channels&quot;: &quot;/**&quot;,
 *      &quot;class&quot;   : &quot;org.mortbay.cometd.filter.NoMarkupFilter&quot;,
 *      &quot;init&quot;    : {}
 *    }
 *  ]
 * </pre>
 * 
 * The init parameter "timeout" specifies the poll timeout in milliseconds
 * (default 45000). The init parameter "multiTimeout" specifies the poll timeout
 * if multiple polls are detected from the same browser (default 0 - disable
 * browser detection).
 * 
 * The init parameter "loglevel" can be set to 0=none, 1=info, 2=debug
 * 
 * If the init parameter "JSONCommented" is true, then the server will accept JSON wrapped
 * in a comment and will generate JSON wrapped in a comment. This is a defence against
 * Ajax Hijacking.
 * 
 * @author gregw
 * @author aabeling: added JSONP transport
 * 
 * @see {@link Bayeux}
 * @see {@link ChannelId}
 */
public abstract class AbstractCometdServlet extends HttpServlet
{
    private static final long serialVersionUID=7376511945687140178L;

    public static final String ORG_MORTBAY_BAYEUX="org.mortbay.bayeux";
    public static final String CLIENT_ATTR="org.mortbay.cometd.client";
    public static final String TRANSPORT_ATTR="org.mortbay.cometd.transport";
    public static final String MESSAGE_PARAM="message";
    public static final String TUNNEL_INIT_PARAM="tunnelInit";
    public static final String HTTP_CLIENT_ID="BAYEUX_HTTP_CLIENT";

    protected Bayeux _bayeux;
    protected long _timeout=45000;

    public Bayeux getBayeux()
    {
        return _bayeux;
    }
    
    protected abstract Bayeux newBayeux();
    
    public void init() throws ServletException
    {
        synchronized (AbstractCometdServlet.class)
        {
            _bayeux=(Bayeux)getServletContext().getAttribute(ORG_MORTBAY_BAYEUX);
            if (_bayeux==null)
            {    
                _bayeux=newBayeux(); 
            }
        }
        
        synchronized(_bayeux)
        {
            boolean was_initialized=_bayeux.isInitialized();
            _bayeux.initialize(getServletContext());
            getServletContext().setAttribute(ORG_MORTBAY_BAYEUX,_bayeux);
            
            if (!was_initialized)
            {
                String filters=getInitParameter("filters");
                if (filters!=null)
                {
                    try
                    {
                        Object[] objects=(Object[])JSON.parse(getServletContext().getResourceAsStream(filters));
                        for (int i=0; objects!=null&&i<objects.length; i++)
                        {
                            Map filter_def=(Map)objects[i];

                            Class c=Thread.currentThread().getContextClassLoader().loadClass((String)filter_def.get("class"));
                            DataFilter filter=(DataFilter)c.newInstance();

                            if (filter instanceof JSONDataFilter)
                                ((JSONDataFilter)filter).init(filter_def.get("init"));

                            _bayeux.addFilter((String)filter_def.get("channels"),filter);
                        }
                    }
                    catch (Exception e)
                    {
                        e.printStackTrace();
                        getServletContext().log("Could not parse: "+filters,e);
                        throw new ServletException(e);
                    }
                }

                String timeout=getInitParameter("timeout");
                if (timeout!=null)
                    _timeout=Long.parseLong(timeout);
                _bayeux.setClientTimeoutMs((_timeout*3)/2);

                String commentedJSON=getInitParameter("JSONCommented");
                _bayeux.setJSONCommented(commentedJSON!=null && Boolean.parseBoolean(commentedJSON));

                String l=getInitParameter("logLevel");
                if (l!=null&&l.length()>0)
                    _bayeux.setLogLevel(Integer.parseInt(l));
            }

        }
    }

    protected void service(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
    {
        String init=req.getParameter(TUNNEL_INIT_PARAM);
        if ("iframe".equals(init))
        {
            throw new IllegalStateException("Not supported");
            /*
            Transport transport=new IFrameTransport();
            ((IFrameTransport)transport).initTunnel(resp);
            */
        }
        else
        {
            super.service(req,resp);
        }
    }

    
}
