// ========================================================================
// Copyright 2006 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at 
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//========================================================================

package org.mortbay.cometd;

import java.io.IOException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.Timer;
import java.util.concurrent.ConcurrentHashMap;
import javax.servlet.ServletContext;

import org.mortbay.thread.Timeout;
import org.mortbay.util.DateCache;
import org.mortbay.util.ajax.JSON;

/* ------------------------------------------------------------ */
/**
 * @author gregw
 * @author aabeling: added JSONP transport
 * 
 */
public class Bayeux
{
    public static final String META_CONNECT="/meta/connect";
    public static final String META_DISCONNECT="/meta/disconnect";
    public static final String META_HANDSHAKE="/meta/handshake";
    public static final String META_PING="/meta/ping";
    public static final String META_RECONNECT="/meta/reconnect";
    public static final String META_STATUS="/meta/status";
    public static final String META_SUBSCRIBE="/meta/subscribe";
    public static final String META_UNSUBSCRIBE="/meta/unsubscribe";

    public static final ChannelId META_CONNECT_ID=new ChannelId(META_CONNECT);
    public static final ChannelId META_DISCONNECT_ID=new ChannelId(META_DISCONNECT);
    public static final ChannelId META_HANDSHAKE_ID=new ChannelId(META_HANDSHAKE);
    public static final ChannelId META_PING_ID=new ChannelId(META_PING);
    public static final ChannelId META_RECONNECT_ID=new ChannelId(META_RECONNECT);
    public static final ChannelId META_STATUS_ID=new ChannelId(META_STATUS);
    public static final ChannelId META_SUBSCRIBE_ID=new ChannelId(META_SUBSCRIBE);
    public static final ChannelId META_UNSUBSCRIBE_ID=new ChannelId(META_UNSUBSCRIBE);
    
    public static final String CLIENT_FIELD="clientId";
    public static final String DATA_FIELD="data";
    public static final String CHANNEL_FIELD="channel";
    public static final String ID_FIELD="id";
    public static final String TIMESTAMP_FIELD="timestamp";
    public static final String TRANSPORT_FIELD="transport";
    public static final String ADVICE_FIELD="advice";
    public static final String SUCCESSFUL_FIELD="successful";
    public static final String SUBSCRIPTION_FIELD="subscription";
    public static final String EXT_FIELD="ext";
    
    private static final JSON.Literal EXT_JSON_COMMENTED=new JSON.Literal("{\"json-comment-filtered\":true}");

    private static HashMap<String,Class> _transports=new HashMap<String,Class>();
    static
    {
        // _transports.put("iframe",IFrameTransport.class);
        _transports.put("long-polling",JSONTransport.class);
        _transports.put("callback-polling",JSONPTransport.class);
    }
    
    HashMap<String,Handler> _handlers=new HashMap<String,Handler>();

    public static final JSON.Literal TRANSPORTS=new JSON.Literal("[\"long-polling\",\"callback-polling\"]");
    private static final JSON.Literal __NO_ADVICE=new JSON.Literal("{}");
    
    Channel _root = new Channel("/",this);
    ConcurrentHashMap<String,Client> _clients=new ConcurrentHashMap<String,Client>();
    SecurityPolicy _securityPolicy=new DefaultPolicy();
    Object _advice=new JSON.Literal("{\"reconnect\":\"retry\",\"interval\":0}");
    Object _unknownAdvice=new JSON.Literal("{\"reconnect\":\"handshake\",\"interval\":500}");
    int _logLevel;
    long _clientTimeoutMs=60000;
    boolean _JSONCommented;
    boolean _initialized;
    

    transient Timer _clientTimer;
    transient ServletContext _context;
    transient Random _random;
    transient DateCache _dateCache;
    transient ConcurrentHashMap<String, ChannelId> _channelIdCache;

    /* ------------------------------------------------------------ */
    /**
     * @param context.
     *            The logLevel init parameter is used to set the logging to
     *            0=none, 1=info, 2=debug
     */
    protected Bayeux()
    {
        _handlers.put("*",new PublishHandler());
        _handlers.put(META_HANDSHAKE,new HandshakeHandler());
        _handlers.put(META_CONNECT,new ConnectHandler());
        _handlers.put(META_RECONNECT,new ReconnectHandler());
        _handlers.put(META_DISCONNECT,new DisconnectHandler());
        _handlers.put(META_SUBSCRIBE,new SubscribeHandler());
        _handlers.put(META_UNSUBSCRIBE,new UnsubscribeHandler());
        _handlers.put(META_STATUS,new StatusHandler());
        _handlers.put(META_PING,new PingHandler());
    }

    /* ------------------------------------------------------------ */
    /**
     * 
     */
    void initialize(ServletContext context)
    {
        synchronized(this)
        {
            _initialized=true;
            _context=context;
            try
            {
                _random=SecureRandom.getInstance("SHA1PRNG");
            }
            catch (Exception e)
            {
                context.log("Could not get secure random for ID generation",e);
                _random=new Random();
            }
            _random.setSeed(_random.nextLong()^hashCode()^(context.hashCode()<<32)^Runtime.getRuntime().freeMemory());
            _dateCache=new DateCache();
            _clientTimer=new Timer();
            _channelIdCache=new ConcurrentHashMap<String, ChannelId>();
        }
    }

    /* ------------------------------------------------------------ */
    public boolean isInitialized()
    {
        return _initialized;
    }

    /* ------------------------------------------------------------ */
    public ChannelId getChannelId(String id)
    {
        ChannelId cid = _channelIdCache.get(id);
        if (cid==null)
        {
            // TODO shrink cache!
            cid=new ChannelId(id);
            _channelIdCache.put(id,cid);
        }
        return cid;
    }
    
    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    /**
     * @param channels
     *            A {@link ChannelId}
     * @param filter
     *            The filter instance to apply to new channels matching the
     *            pattern
     */
    public void addFilter(String channels, DataFilter filter)
    {
        synchronized (this)
        {
            Channel channel = getChannel(channels,true);
            channel.addDataFilter(filter);
        }
    }

    /* ------------------------------------------------------------ */
    /**
     * @param id
     * @return
     */
    public Channel getChannel(String id)
    {
        ChannelId cid=getChannelId(id);
        if (cid.depth()==0)
            return null;
        return _root.getChild(cid);
    }


    /* ------------------------------------------------------------ */
    /**
     * @param id
     * @return
     */
    public Channel getChannel(ChannelId id)
    {
        return _root.getChild(id);
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    /**
     * @param id
     * @param create TODO
     * @return
     */
    public Channel getChannel(String id, boolean create)
    {
        synchronized(this)
        {
            Channel channel=getChannel(id);

            if (channel==null && create)
            {
                channel=new Channel(id,this);
                _root.addChild(channel);
                
                if (isLogInfo())
                    logInfo("newChannel: "+channel);
            }
            return channel;
        }
    }

    /* ------------------------------------------------------------ */
    /**
     * @param client_id
     * @return
     */
    public Client getClient(String client_id)
    {
        synchronized(this)
        {
            if (client_id==null)
                return null;
            Client client = _clients.get(client_id);
            return client;
        }
    }

    /* ------------------------------------------------------------ */
    /**
     * @param client_id
     */
    public Client removeClient(String client_id)
    {
        Client client;
        synchronized(this)
        {
            if (client_id==null)
                return null;
            client = _clients.remove(client_id);
        }
        if (client!=null)
        {
            client.unsubscribeAll();
        }
        return client;
    }
    
    /* ------------------------------------------------------------ */
    /** Construct a new {@link Client} instance. 
     * @param idPrefix Prefix to random client ID
     * @return new Client
     */
    protected Client newClient(String idPrefix)
    {
        return new Client(this,idPrefix);
    }

    /* ------------------------------------------------------------ */
    /**
     * @return
     */
    public Set getClientIDs()
    {
        return _clients.keySet();
    }

    /* ------------------------------------------------------------ */
    /**
     * @return
     */
    String getTimeOnServer()
    {
        return _dateCache.format(System.currentTimeMillis());
    }

    /* ------------------------------------------------------------ */
    /**
     * @param client
     * @param message
     * @return
     */
    public Transport newTransport(Client client, Map message)
    {
        if (isLogDebug())
            logDebug("newTransport: client="+client+",message="+message);

        Transport result=null;

        try
        {
            String type=client==null?null:client.getConnectionType();
            if (type==null)
                type=(String)message.get("connectionType");

            /* check if jsonp parameter is set */
            if (type==null)
            {
                String jsonp=(String)message.get("jsonp");
                if (jsonp!=null)
                {
                    /* use JSONPTransport */
                    if (isLogDebug())
                        logDebug("newTransport: using JSONPTransport with jsonp="+jsonp);
                    result=new JSONPTransport(client!=null&&client.isJSONCommented());
                    ((JSONPTransport)result).setJsonp(jsonp);
                }
            }

            if ((type!=null)&&(result==null))
            {
                Class trans_class=(Class)_transports.get(type);
                if (trans_class!=null)
                {
                    if (trans_class.equals(JSONPTransport.class))
                    {
                        String jsonp=(String)message.get("jsonp");
                        if (jsonp==null)
                        {
                            throw new Exception("JSONPTransport needs jsonp parameter");
                        }
                        result=new JSONPTransport(client!=null&&client.isJSONCommented());
                        ((JSONPTransport)result).setJsonp(jsonp);
                    }
                    else
                    {
                        result=(Transport)(trans_class.newInstance());
                        result.setJSONCommented(client!=null&&client.isJSONCommented());
                    }
                }
            }

            if (result==null)
            {
                result=new JSONTransport(client!=null&&client.isJSONCommented());
            }
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }

        if (isLogDebug())
            logDebug("newTransport: result="+result);
        return result;
    }

    /* ------------------------------------------------------------ */
    /** Handle a Bayeux request
     * @param client
     * @param transport
     * @param message
     * @return
     */
    public void handle(Client client, Transport transport, Map<String,Object> message) throws IOException
    {
        final String METHOD="handle: ";

        String channel_id=(String)message.get(CHANNEL_FIELD);

        Handler handler=(Handler)_handlers.get(channel_id);
        if (handler==null)
            handler=(Handler)_handlers.get("*");

        if (isLogDebug())
            logDebug(METHOD+"handler="+handler);

        handler.handle(client,transport,message);
    }


    /* ------------------------------------------------------------ */
    public void publish(ChannelId to, Client from, Object data, String msgId)
    {
        HashMap<String,Object> msg = new HashMap<String,Object>();
        msg.put(Bayeux.CHANNEL_FIELD,to.toString());
        msg.put(Bayeux.TIMESTAMP_FIELD,getTimeOnServer());
        if (msgId==null)
        {
            long id=msg.hashCode()
            ^(to==null?0:to.hashCode())
            ^(from==null?0:from.hashCode());
            id=id<0?-id:id;
            msg.put(Bayeux.ID_FIELD,Long.toString(id,36));
        }
        msg.put(Bayeux.DATA_FIELD,data);
        _root.publish(to,from,msg);
    }
    
    /* ------------------------------------------------------------ */
    void advise(Client client, Transport transport, Object advice) throws IOException
    {
        if (advice==null)
            advice=_advice;
        if (advice==null)
            advice=__NO_ADVICE;
        String channel="/meta/connections/"+client.getId();
        Map<String,Object> reply=new HashMap<String,Object>();
        reply.put(CHANNEL_FIELD,channel);
        reply.put("timestamp",_dateCache.format(System.currentTimeMillis()));
        reply.put(SUCCESSFUL_FIELD,Boolean.TRUE);
        reply.put(ADVICE_FIELD,advice);
        transport.send(reply);
    }

    /* ------------------------------------------------------------ */
    long getRandom(long variation)
    {
        long l=_random.nextLong()^variation;
        return l<0?-l:l;
    }

    /* ------------------------------------------------------------ */
    public SecurityPolicy getSecurityPolicy()
    {
        return _securityPolicy;
    }

    /* ------------------------------------------------------------ */
    public void setSecurityPolicy(SecurityPolicy securityPolicy)
    {
        _securityPolicy=securityPolicy;
    }

    /* ------------------------------------------------------------ */
    /**
     * @return the logLevel. 0=none, 1=info, 2=debug
     */
    public int getLogLevel()
    {
        return _logLevel;
    }

    /* ------------------------------------------------------------ */
    /**
     * @param logLevel
     *            the logLevel: 0=none, 1=info, 2=debug
     */
    public void setLogLevel(int logLevel)
    {
        _logLevel=logLevel;
    }

    /* ------------------------------------------------------------ */
    public boolean isLogInfo()
    {
        return _logLevel>0;
    }

    /* ------------------------------------------------------------ */
    public boolean isLogDebug()
    {
        return _logLevel>1;
    }

    /* ------------------------------------------------------------ */
    public void logInfo(String message)
    {
        if (_logLevel>0)
            _context.log(message);
    }

    /* ------------------------------------------------------------ */
    public void logDebug(String message)
    {
        if (_logLevel>1)
            _context.log(message);
    }

    /* ------------------------------------------------------------ */
    public void logDebug(String message, Throwable th)
    {
        if (_logLevel>1)
            _context.log(message,th);
    }

    /* ------------------------------------------------------------ */
    public long getClientTimeoutMs()
    {
        return _clientTimeoutMs;
    }

    /* ------------------------------------------------------------ */
    public void setClientTimeoutMs(long ms)
    {
        _clientTimeoutMs=ms;
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private interface Handler
    {
        void handle(Client client, Transport transport, Map<String, Object> message) throws IOException;
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class ConnectHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
            final String METHOD="handle: ";

            if (client==null)
                throw new IllegalStateException("No client");
            
            Map<String,Object> reply=new HashMap<String,Object>();
            reply.put(CHANNEL_FIELD,META_CONNECT);

            String type=(String)message.get("connectionType");
            client.setConnectionType(type);
            if (isLogDebug())
                logDebug(METHOD+"connectionType set to "+type);

            Channel connection=client.connect();
            if (connection!=null)
            {
                reply.put(SUCCESSFUL_FIELD,Boolean.TRUE);
                reply.put("error","");
            }
            else
            {
                reply.put(SUCCESSFUL_FIELD,Boolean.FALSE);
                reply.put("error","unknown client ID");
                if (_unknownAdvice!=null)
                    reply.put(ADVICE_FIELD,_unknownAdvice);
            }
            reply.put("timestamp",_dateCache.format(System.currentTimeMillis()));
            transport.send(reply);
            transport.setPolling(false);
            
            _root.publish(META_CONNECT_ID,client,reply);
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class PublishHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
            String channel_id=(String)message.get(CHANNEL_FIELD);
            String id=(String)message.get(ID_FIELD);

            ChannelId cid=getChannelId(channel_id);
            Object data=message.get("data");

            if (client==null)
            {
                if (_securityPolicy.authenticate((String)message.get("authScheme"),(String)message.get("authUser"),(String)message.get("authToken")))
                    client=newClient(null);
            }

            Map<String,Object> reply=new HashMap<String,Object>();
            reply.put(CHANNEL_FIELD,channel_id);
            
            if (id!=null)
                reply.put(ID_FIELD,id);
                
            if (data!=null&&_securityPolicy.canSend(client,cid,message))
            {
                publish(cid,client,data,id);
                reply.put(SUCCESSFUL_FIELD,Boolean.TRUE);
                reply.put("error","");
            }
            else
            {
                reply.put(SUCCESSFUL_FIELD,Boolean.FALSE);
                reply.put("error","unknown channel");
            }
            transport.send(reply);
        }

    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class DisconnectHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message)
        {
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class HandshakeHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
            if (isLogDebug())
                logDebug("handshake.handle: client="+client+",transport="+transport+",message="+message);

            if (client!=null)
                throw new IllegalStateException();

            _root.publish(META_HANDSHAKE_ID,client,message);
            
            if (_securityPolicy.authenticate((String)message.get("authScheme"),(String)message.get("authUser"),(String)message.get("authToken")))
                client=newClient(null);

            Map ext = (Map)message.get(EXT_FIELD);

            boolean commented=_JSONCommented && ext!=null && ((Boolean)ext.get("json-comment-filtered")).booleanValue();
            
            Map<String,Object> reply=new HashMap<String,Object>();
            reply.put(CHANNEL_FIELD,META_HANDSHAKE);
            reply.put("version",new Double(0.1));
            reply.put("minimumVersion",new Double(0.1));
            if (isJSONCommented())
                reply.put(EXT_FIELD,EXT_JSON_COMMENTED);

            if (client!=null)
            {
                reply.put("supportedConnectionTypes",TRANSPORTS);
                reply.put("authSuccessful",Boolean.TRUE);
                reply.put(CLIENT_FIELD,client.getId());
                if (_advice!=null)
                    reply.put(ADVICE_FIELD,_advice);
                client.setJSONCommented(commented);
                transport.setJSONCommented(commented);
            }
            else
            {
                reply.put("authSuccessful",Boolean.FALSE);
                if (_advice!=null)
                    reply.put(ADVICE_FIELD,_advice);
            }

            if (isLogDebug())
                logDebug("handshake.handle: reply="+reply);

            transport.send(reply);
            _root.publish(META_HANDSHAKE_ID,client,reply);
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class PingHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class ReconnectHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
            _root.publish(META_RECONNECT_ID,client,message);
            
            Map<String,Object> reply=new HashMap<String,Object>();
            reply.put(CHANNEL_FIELD,META_RECONNECT);
            reply.put("timestamp",_dateCache.format(System.currentTimeMillis()));

            if (client==null)
            {
                reply.put(SUCCESSFUL_FIELD,Boolean.FALSE);
                reply.put("error","unknown clientID");
                if (_unknownAdvice!=null)
                    reply.put(ADVICE_FIELD,_unknownAdvice);
                transport.setPolling(false);
                transport.send(reply);
            }
            else
            {
                String type=(String)message.get("connectionType");
                if (type!=null)
                {
                    /*
                     * aabeling: workaround to remain on callback-polling even
                     * if dojo client advises another connection type
                     */
                    if (isLogDebug())
                        logDebug("Reconnect.handle: old connectionType="+client.getConnectionType());
                    if (client.getConnectionType().equals("callback-polling"))
                    {
                        if (isLogDebug())
                            logDebug("Reconnect.handle: connectionType remains callback-polling");
                    }
                    else
                    {
                        client.setConnectionType(type);
                        if (isLogDebug())
                            logDebug("Reconnect.handle: connectionType reset to "+type);
                    }
                }
                reply.put(SUCCESSFUL_FIELD,Boolean.TRUE);
                reply.put("error","");
                transport.setPolling(true);
                transport.send(reply);
            }

            _root.publish(META_RECONNECT_ID,client,reply);
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class StatusHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class SubscribeHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
            if (client==null)
                throw new IllegalStateException("No client");

            _root.publish(META_SUBSCRIBE_ID,client,message);
            
            String subscribe_id=(String)message.get(SUBSCRIPTION_FIELD);

            // select a random channel ID if none specifified
            if (subscribe_id==null)
            {
                subscribe_id=Long.toString(getRandom(message.hashCode()^client.hashCode()),36);
                while (getChannel(subscribe_id)!=null)
                    subscribe_id=Long.toString(getRandom(message.hashCode()^client.hashCode()),36);
            }

            ChannelId cid=getChannelId(subscribe_id);
            
            Map<String,Object> reply=new HashMap<String,Object>();
            reply.put(CHANNEL_FIELD,META_SUBSCRIBE);
            reply.put(SUBSCRIPTION_FIELD,subscribe_id);

            if (_securityPolicy.canSubscribe(client,cid,message))
            {
                Channel channel=getChannel(cid);
                if (channel==null&&_securityPolicy.canCreate(client,cid,message))
                    channel=getChannel(subscribe_id, true);
                
                if (channel!=null)
                {
                    channel.addSubscriber(client);
                    reply.put(SUCCESSFUL_FIELD,Boolean.TRUE);
                    reply.put("error","");
                }
                else 
                {
                    reply.put(SUCCESSFUL_FIELD,Boolean.FALSE);
                    reply.put("error","cannot create");
                }
            }
            else
            {
                reply.put(SUCCESSFUL_FIELD,Boolean.FALSE);
                reply.put("error","cannot subscribe");
                
            }
            
            transport.send(reply);

            _root.publish(META_SUBSCRIBE_ID,client,reply);

        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private class UnsubscribeHandler implements Handler
    {
        public void handle(Client client, Transport transport, Map<String, Object> message) throws IOException
        {
            if (client==null)
                return;

            _root.publish(META_UNSUBSCRIBE_ID,client,message);
            String channel_id=(String)message.get(SUBSCRIPTION_FIELD);
            Channel channel=getChannel(channel_id);
            if (channel!=null)
                channel.removeSubscriber(client);

            Map<String,Object> reply=new HashMap<String,Object>();
            reply.put(CHANNEL_FIELD,channel_id);
            reply.put(SUBSCRIPTION_FIELD,channel.getId());
            reply.put(SUCCESSFUL_FIELD,Boolean.TRUE);
            reply.put("error","");
            transport.send(reply);
            _root.publish(META_UNSUBSCRIBE_ID,client,reply);
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    private static class DefaultPolicy implements SecurityPolicy
    {

        public boolean canCreate(Client client, ChannelId channel, Map message)
        {
            return client!=null && !"meta".equals(channel.getSegment(0));
        }

        public boolean canSubscribe(Client client, ChannelId channel, Map message)
        {
            return client!=null && !"meta".equals(channel.getSegment(0));
        }

        public boolean canSend(Client client, ChannelId channel, Map message)
        {
            return client!=null && !"meta".equals(channel.getSegment(0));
        }

        public boolean authenticate(String scheme, String user, String credentials)
        {
            return true;
        }

    }

    /* ------------------------------------------------------------ */
    /**
     * @return the commented
     */
    public boolean isJSONCommented()
    {
        return _JSONCommented;
    }

    /* ------------------------------------------------------------ */
    /**
     * @param commented the commented to set
     */
    public void setJSONCommented(boolean commented)
    {
        _JSONCommented=commented;
    }

}
