package org.mortbay.cometd.filter;

import java.lang.reflect.Array;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.mortbay.cometd.Channel;
import org.mortbay.cometd.Client;
import org.mortbay.cometd.DataFilter;
import org.mortbay.util.LazyList;
import org.mortbay.util.ajax.JSON;

/** JSON DataFilter
 * This {@link DataFilter} walks an Object as if it was a call to {@link JSON#toString(Object)} and 
 * calls the protected methods 
 * {@link #filterString(String)},
 * {@link #filterNumber(Number)},
 * {@link #filterBoolean(Boolean)},
 * {@link #filterArray(Object, Client)} or
 * {@link #filterMap(Map, Client)} appropriate.
 * Derived filters may override one or more of these methods to provide filtering of specific types.
 * 
 * @author gregw
 *
 */
public class JSONDataFilter implements DataFilter
{
    public void init(Object init)
    {}
    
    public Object filter(Client from, Channel to, Object data) throws IllegalStateException
    {
        if (data==null)
            return null;
        
        if (data instanceof Map)
            return filterMap(from,to,(Map)data);
        if (data instanceof List)
            return filterArray(from,to,LazyList.toArray(data,Object.class));
        if (data.getClass().isArray() )
            return filterArray(from,to,data);
        if (data instanceof Number)
            return filterNumber((Number)data);
        if (data instanceof Boolean)
            return filterBoolean((Boolean)data);
        if (data instanceof String)
            return filterString((String)data);
        return filterString(data.toString());
    }

    protected Object filterString(String string)
    {
        return string;
    }

    protected Object filterBoolean(Boolean bool)
    {
        return bool;
    }

    protected Object filterNumber(Number number)
    {
        return number;
    }

    protected Object filterArray(Client from, Channel to, Object array)
    {
        if (array==null)
            return null;
        
        int length = Array.getLength(array);
        
        for (int i=0;i<length;i++)
            Array.set(array,i,filter(from, to, Array.get(array,i)));
        
        return array;
    }

    protected Object filterMap(Client from, Channel to, Map object)
    {
        if (object==null)
            return null;
        
        Iterator iter = object.entrySet().iterator();
        while(iter.hasNext())
        {
            Map.Entry entry = (Map.Entry)iter.next();
            entry.setValue(filter(from, to, entry.getValue()));
        }

        return object;
    }

}
