// ========================================================================
// Copyright 2006 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at 
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ========================================================================

package org.mortbay.cometd;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.mortbay.log.Log;
import org.mortbay.util.LazyList;

/* ------------------------------------------------------------ */
/** A Bayuex Channel
 * 
 * @author gregw
 *
 */
public class Channel
{
    private Bayeux _bayeux;
    private Client[] _subscribers=new Client[0]; // copy on write
    private DataFilter[] _dataFilters=new DataFilter[0]; // copy on write
    private ChannelId _id;
    private ConcurrentMap<String,Channel> _children = new ConcurrentHashMap<String, Channel>();
    private Channel _wild;
    private Channel _wildWild;

    /* ------------------------------------------------------------ */
    Channel(String id,Bayeux bayeux)
    {
        _id=new ChannelId(id);
        _bayeux=bayeux;
    }

    /* ------------------------------------------------------------ */
    public void addChild(Channel channel)
    {
        ChannelId child=channel.getId();
        if (!_id.isParentOf(child))
        {
            throw new IllegalArgumentException(_id+" not parent of "+child);
        }
        
        String next = child.getSegment(_id.depth());

        if ((child.depth()-_id.depth())==1)
        {
            // add the channel to this channels
            Channel old = _children.putIfAbsent(next,channel);

            if (old!=null)
                throw new IllegalArgumentException("Already Exists");

            if (ChannelId.WILD.equals(next))
                _wild=channel;
            else if (ChannelId.WILDWILD.equals(next))
                _wildWild=channel;
                
        }
        else
        {
            Channel branch=_children.get(next);
                branch=_bayeux.getChannel((_id.depth()==0?"/":(_id.toString()+"/"))+next,true);
            
            branch.addChild(channel);
        }
    }
    
    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    /**
     * @param client
     */
    public void addSubscriber(Client client)
    {
        client.addSubscription(this);
        synchronized (this)
        {
            _subscribers=(Client[])LazyList.addToArray(_subscribers,client,null);
        }
    }

    /* ------------------------------------------------------------ */
    /**
     * @param client
     */
    public void removeSubscriber(Client client)
    {
        client.removeSubscription(this);
        synchronized(this)
        {
            _subscribers=(Client[])LazyList.removeFromArray(_subscribers,client);
        }
    }
    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    /**
     * @param filter
     */
    public void addDataFilter(DataFilter filter)
    {
        synchronized(this)
        {
            _dataFilters=(DataFilter[])LazyList.addToArray(_dataFilters,filter,null);
        }
    }

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    /**
     * @param filter
     */
    public void removeDataFilter(DataFilter filter)
    {
        synchronized(this)
        {
            _dataFilters=(DataFilter[])LazyList.removeFromArray(_dataFilters,filter);
        }
    }
    

    /* ------------------------------------------------------------ */
    /* ------------------------------------------------------------ */
    /**
     * @return
     */
    public ChannelId getId()
    {
        return _id;
    }


    /* ------------------------------------------------------------ */
    public void publish(ChannelId to, Client from, Map<String,Object> msg)
    {
        int tail = to.depth()-_id.depth();
        
        Object data = msg.get(Bayeux.DATA_FIELD);
        Object old = data;
        
        DataFilter[] filters=null;
        
        try
        {
            switch(tail)
            {
                case 0:      
                {
                    synchronized(this)
                    {
                        filters=_dataFilters;
                    }
                    for (DataFilter filter: filters)
                        data=filter.filter(from,this,data);
                }
                break;

                case 1:
                    if (_wild!=null)  
                    {
                        synchronized(_wild)
                        {
                            filters=_wild._dataFilters;
                        }
                        for (DataFilter filter: filters)
                            data=filter.filter(from,this,data);
                    }

                default:
                    if (_wildWild!=null)  
                    {
                        synchronized(_wildWild)
                        {
                            filters=_wildWild._dataFilters;
                        }
                        for (DataFilter filter: filters)
                        {
                            data=filter.filter(from,this,data);
                        }
                    }
            }
        }
        catch (IllegalStateException e)
        {
            Log.debug(e);
            return;
        }
        if (data!=old)
            msg.put(Bayeux.DATA_FIELD,data);
        
        boolean delivered=false;
        Client[] subscribers;

        switch(tail)
        {
            case 0:
                synchronized (this)
                {
                    subscribers=_subscribers;
                }
                for (Client client: subscribers)
                {
                    client.deliver(from,to,msg);
                    delivered=true;
                }
                
                break;

            case 1:

                if (_wild!=null)
                {
                    synchronized (_wild)
                    {
                        subscribers=_wild._subscribers;
                    }
                    for (Client client: subscribers)
                    {
                        client.deliver(from,to,msg);
                        delivered=true;
                    }
                }

            default:
            {
                if (_wildWild!=null)
                {
                    synchronized (_wildWild)
                    {
                        subscribers=_wildWild._subscribers;
                    }
                    for (Client client: subscribers)
                    {
                        client.deliver(from,to,msg);
                        delivered=true;
                    }
                }
                String next = to.getSegment(_id.depth());
                Channel channel = _children.get(next);
                if (channel!=null)
                {
                    if (delivered)
                    {
                        msg=new HashMap<String,Object>(msg);
                    }
                    channel.publish(to,from,msg);
                }
            }
        }
    }
    
    /* ------------------------------------------------------------ */
    /**
     * @param client The client for which this token will be valid
     * @param subscribe True if this token may be used for subscriptions
     * @param send True if this token may be used for send
     * @param oneTime True if this token may only be used in one request batch.
     * @return A new token that can be used for subcriptions and or sending.
     */
    public String getToken(Client client, boolean subscribe, boolean send, boolean oneTime)
    {
        String token=Long.toString(_bayeux.getRandom(client.hashCode()),36);
        // TODO register somewhere ?
        return token;
    }
    
    /* ------------------------------------------------------------ */
    public String toString()
    {
        return _id.toString();
    }

    /* ------------------------------------------------------------ */
    public Channel getChild(ChannelId id)
    {
        String next=id.getSegment(_id.depth());
        if (next==null)
            return null;
        
        Channel channel = _children.get(next);
        
        if (channel==null || channel.getId().depth()==id.depth())
        {
            return channel;
        }
        return channel.getChild(id);
    }
}
