// ========================================================================
// Copyright 2006 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at 
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//========================================================================

package org.mortbay.cometd;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.atomic.AtomicInteger;

import org.mortbay.util.LazyList;


/* ------------------------------------------------------------ */
/**
 * 
 * @author gregw
 */
public class Client
{
    protected Bayeux _bayeux;
    private String _id;
    private String _type;
    private Queue<Map<String,Object>> _messageQ = new LinkedList<Map<String,Object>>();
    private AtomicInteger _responsesPending = new AtomicInteger(0);
    private Channel _connection=null;
    private Channel[] _subscriptions=new Channel[0]; // copy of write
    private long _accessed;
    private transient TimerTask _timeout; 
    private boolean _JSONCommented;
    
    /* ------------------------------------------------------------ */
    public Client(Bayeux bayeux, String idPrefix)
    {
        _bayeux=bayeux;
        if (idPrefix==null)
            _id=Long.toString(bayeux.getRandom(System.identityHashCode(this)^System.currentTimeMillis()),36);
        else
            _id=idPrefix+"_"+Long.toString(bayeux.getRandom(System.identityHashCode(this)^System.currentTimeMillis()),36);
        
        _bayeux._clients.put(getId(),this);
        if (_bayeux.isLogInfo())
            _bayeux.logInfo("newClient: "+this);
    }

    /* ------------------------------------------------------------ */
    /**
     * @return the commented
     */
    public boolean isJSONCommented()
    {
        synchronized(this)
        {
            return _JSONCommented;
        }
    }

    /* ------------------------------------------------------------ */
    /**
     * @param commented the commented to set
     */
    public void setJSONCommented(boolean commented)
    {
        synchronized(this)
        {
            _JSONCommented=commented;
        }
    }

    /* ------------------------------------------------------------ */
    public void remove()
    {
        _bayeux.removeClient(_id);        
    }
    
    /* ------------------------------------------------------------ */
    public String getConnectionType()
    {
        return _type;
    }
    
    /* ------------------------------------------------------------ */
    public String getId()
    {
        return _id;
    }
    
    /* ------------------------------------------------------------ */
    /**
     * Connect the client.
     * @return the meta Channel for the connection
     */
    public Channel connect()
    {
        synchronized (this)
        {
            String connection_id="/meta/connections/"+getId();
            _connection=_bayeux.getChannel(connection_id, true);
            _connection.addSubscriber(this);
            return _connection;
        }
    }
   
    /* ------------------------------------------------------------ */
    /**
     * @return the meta Channel for the connection or null if not connected.
     */
    public Channel getConnection()
    {
        return _connection;
    }
    
    /* ------------------------------------------------------------ */
    public boolean hasMessages()
    {
        synchronized(this)
        {
            return _messageQ.size()>0;
        }
    }
    
    /* ------------------------------------------------------------ */
    public Queue<Map<String,Object>> takeMessages()
    {
        synchronized(this)
        {
            if (_messageQ.size()==0)
                return null;
            
            Queue<Map<String,Object>> messages = _messageQ;
            _messageQ=new LinkedList<Map<String,Object>>();
            return messages;
        }
    }
       
    /* ------------------------------------------------------------ */
    public String toString()
    {
        return _id;
    }
    
    /* ------------------------------------------------------------ */
    protected void deliver(Client from, ChannelId to, Map<String,Object> message)
    {
        synchronized(this)
        {
            if (_connection!=null)
            {
                _messageQ.add(message);
            
                if (_responsesPending.get()<1)
                    resume();
            }
        }
    }

    /* ------------------------------------------------------------ */
    /** Called by deliver to resume anything waiting on this client.
     */
    public void resume()
    {
    }
    
    /* ------------------------------------------------------------ */
    protected int responded()
    {
        return _responsesPending.getAndDecrement();
    }

    /* ------------------------------------------------------------ */
    protected int responsePending()
    {
        return _responsesPending.incrementAndGet();
    }
    
    /* ------------------------------------------------------------ */
    protected void setConnectionType(String type)
    {
        synchronized (this)
        {
            _type=type;
        }
    }

    /* ------------------------------------------------------------ */
    protected void setId(String _id)
    {
        synchronized (this)
        {
            this._id=_id;
        }
    }

    /* ------------------------------------------------------------ */
    protected void addSubscription(Channel channel)
    {
        synchronized (this)
        {
            _subscriptions=(Channel[])LazyList.addToArray(_subscriptions,channel,null);
        }
    }

    /* ------------------------------------------------------------ */
    protected void removeSubscription(Channel channel)
    {
        synchronized (this)
        {
            _subscriptions=(Channel[])LazyList.removeFromArray(_subscriptions,channel);
        }
    }

    /* ------------------------------------------------------------ */
    protected void access()
    {
        synchronized(this)
        {
            // distribute access time in cluster
            _accessed=System.currentTimeMillis();
        
            if (_timeout!=null)
                _timeout.cancel();
            
            // a bit of a waste creating a new instance every time!
            _timeout = new TimerTask()
            {
                public void run()
                {
                    long now = System.currentTimeMillis();
                    if (_accessed+_bayeux.getClientTimeoutMs()<now)
                        remove();
                }
            };
                
            _bayeux._clientTimer.schedule(_timeout,_bayeux.getClientTimeoutMs());
        }
        
    }

    /* ------------------------------------------------------------ */
    protected void unsubscribeAll()
    {
        Channel[] subscriptions;
        synchronized(this)
        {
            _messageQ.clear();
            subscriptions=_subscriptions;
            _subscriptions=new Channel[0];
        }
        for (Channel channel : subscriptions)
            channel.removeSubscriber(this);
        
    }

}