package org.mortbay.jetty.servlet.wadi;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.catalina.tribes.Member;
import org.codehaus.wadi.core.assembler.StackContext;
import org.codehaus.wadi.core.contextualiser.Contextualiser;
import org.codehaus.wadi.core.manager.Manager;
import org.codehaus.wadi.core.manager.SessionAlreadyExistException;
import org.codehaus.wadi.core.manager.SessionMonitor;
import org.codehaus.wadi.core.store.Store;
import org.codehaus.wadi.group.Dispatcher;
import org.codehaus.wadi.replication.manager.ReplicationManagerFactory;
import org.codehaus.wadi.replication.manager.basic.BasicReplicationManagerFactory;
import org.codehaus.wadi.replication.manager.basic.NoOpReplicationManagerFactory;
import org.codehaus.wadi.replication.storage.ReplicaStorageFactory;
import org.codehaus.wadi.replication.storage.basic.BasicReplicaStorageFactory;
import org.codehaus.wadi.replication.strategy.BackingStrategyFactory;
import org.codehaus.wadi.replication.strategy.RoundRobinBackingStrategyFactory;
import org.codehaus.wadi.servicespace.ServiceSpace;
import org.codehaus.wadi.servicespace.ServiceSpaceName;
import org.codehaus.wadi.tribes.TribesDispatcher;
import org.codehaus.wadi.web.impl.URIEndPoint;
import org.mortbay.jetty.servlet.AbstractSessionManager;
import org.mortbay.jetty.servlet.HashSessionIdManager;
import org.mortbay.log.Log;

import edu.emory.mathcs.backport.java.util.concurrent.CopyOnWriteArrayList;

public class WadiSessionManager extends AbstractSessionManager
{
    private final Map<String, ClusteredSession> _idToSession = new HashMap<String, ClusteredSession>();

    private ReplicationManagerFactory _repManagerFactory;

    private ReplicaStorageFactory _repStorageFactory;

    private BackingStrategyFactory _backingStrategyFactory;

    private Dispatcher _dispatcher;

    private Collection<Member> _staticMembers;
    
    private CopyOnWriteArrayList _listeners;

    private Manager _manager;

    private SessionMonitor _sessionMonitor;

    private ServiceSpace _serviceSpace;

    private String _clusterName;

    private URI _serviceSpaceName;

    private String _nodeName;

    private int _nbReplica;

    private URI _endPointURI;

    private int _numPartitions;

    private int _sweepInterval;

    private final boolean _enableReplication;
    private Store _sharedStore;
    
    public WadiSessionManager(String clusterName,
            String serviceSpaceName,
            String nodeName,
            String endPoint,
            int numPartitions,
            int sweepInterval)
    {
        this(clusterName, serviceSpaceName, nodeName, 0, endPoint, numPartitions, sweepInterval, false);
    }

    public WadiSessionManager(String clusterName,
            String serviceSpaceName,
            String nodeName,
            int nbReplica,
            String endPoint,
            int numPartitions,
            int sweepInterval)
    {
        this(clusterName, serviceSpaceName, nodeName, nbReplica, endPoint, numPartitions, sweepInterval, true);
    }
    
    protected WadiSessionManager(String clusterName,
            String serviceSpaceName,
            String nodeName,
            int nbReplica,
            String endPoint,
            int numPartitions,
            int sweepInterval,
            boolean enableReplication)
    {
        this._clusterName = clusterName;
        this._nodeName = nodeName;
        this._nbReplica = nbReplica;
        this._enableReplication = enableReplication;
        try
        {
            this._serviceSpaceName = new URI(serviceSpaceName);
            this._endPointURI = new URI(endPoint);
        }
        catch (URISyntaxException e)
        {
            Log.warn(e);
        }
        this._numPartitions = numPartitions;
        this._sweepInterval = sweepInterval;
        _listeners = new CopyOnWriteArrayList();

        HashSessionIdManager sessionIdManager = new HashSessionIdManager();
        sessionIdManager.setWorkerName(nodeName);
        setIdManager(sessionIdManager);

        registerListener(new MigrationListener());
    }

    public Store getSharedStore()
    {
        return _sharedStore;
    }

    public void setSharedStore(Store store)
    {
        _sharedStore = store;
    }

    public void addStaticMember(Member member)
    {
        if (_staticMembers == null) _staticMembers = new ArrayList<Member>();
        _staticMembers.add(member);
    }
    
    public void doStart() throws Exception
    {
        super.doStart();
        _repManagerFactory = new NoOpReplicationManagerFactory();
        if (_enableReplication) {
            _repManagerFactory = new BasicReplicationManagerFactory();
        }
        _repStorageFactory = new BasicReplicaStorageFactory();
        _backingStrategyFactory = new RoundRobinBackingStrategyFactory(_nbReplica);
        
        _dispatcher = newDispatcher();
        
        _dispatcher.start();
        
        ServiceSpaceName serviceSpaceName = new ServiceSpaceName(_serviceSpaceName);
        StackContext stackContext = new StackContext(serviceSpaceName,
                _dispatcher,
                _dftMaxIdleSecs,
                _numPartitions,
                _sweepInterval,
                _repManagerFactory,
                _repStorageFactory,
                _backingStrategyFactory) {
            @Override
            protected Store getSharedStore() {
                return _sharedStore;
            }
            @Override
            protected Contextualiser newReplicaAwareContextualiser(Contextualiser next) {
                if (_enableReplication) {
                    return super.newReplicaAwareContextualiser(next);
                } else {
                    return next;
                }
            }
        };
        stackContext.build();

        _serviceSpace = stackContext.getServiceSpace();
        _manager = stackContext.getManager();

        _sessionMonitor = stackContext.getSessionMonitor();
        _sessionMonitor.addSessionListener(new SessionListenerAdapter());

        _serviceSpace.start();
    }

    private Dispatcher newDispatcher()
    {
        if (_staticMembers != null)
            return new TribesDispatcher(_clusterName, _nodeName, new URIEndPoint(_endPointURI), _dftMaxIdleSecs, null, _staticMembers);
        else
            return new TribesDispatcher(_clusterName, _nodeName, new URIEndPoint(_endPointURI), _dftMaxIdleSecs, null);
    }
    
    public void doStop() throws Exception
    {
        _serviceSpace.stop();
        super.doStop();
    }

    public Manager getClusteredManager()
    {
        return _manager;
    }

    public WadiSession createSession(String sessionId) throws SessionAlreadyExistsException
    {
        org.codehaus.wadi.core.session.Session session;
        try
        {
            session = _manager.createWithName(sessionId);
        }
        catch (SessionAlreadyExistException e)
        {
            throw new SessionAlreadyExistsException(sessionId);
        }
        return new WadiSessionAdaptor(session);
    }

    public void registerListener(SessionListener listener)
    {
        _listeners.add(listener);
    }

    public void unregisterListener(SessionListener listener)
    {
        _listeners.remove(listener);
    }

    private void notifyInboundSessionMigration(org.codehaus.wadi.core.session.Session session)
    {
        for (Iterator iter = _listeners.iterator(); iter.hasNext();)
        {
            SessionListener listener = (SessionListener) iter.next();
            listener.notifyInboundSessionMigration(new WadiSessionAdaptor(session));
        }
    }

    private void notifyOutboundSessionMigration(org.codehaus.wadi.core.session.Session session)
    {
        for (Iterator iter = _listeners.iterator(); iter.hasNext();)
        {
            SessionListener listener = (SessionListener) iter.next();
            listener.notifyOutboundSessionMigration(new WadiSessionAdaptor(session));
        }
    }

    private void notifySessionDestruction(org.codehaus.wadi.core.session.Session session)
    {
        for (Iterator iter = _listeners.iterator(); iter.hasNext();)
        {
            SessionListener listener = (SessionListener) iter.next();
            listener.notifySessionDestruction(new WadiSessionAdaptor(session));
        }
    }

    private class SessionListenerAdapter implements org.codehaus.wadi.core.manager.SessionListener
    {

        public void onSessionCreation(org.codehaus.wadi.core.session.Session session) {
        }

        public void onSessionDestruction(org.codehaus.wadi.core.session.Session session) {
            notifySessionDestruction(session);
        }

        public void onInboundSessionMigration(org.codehaus.wadi.core.session.Session session) {
            notifyInboundSessionMigration(session);
        }
        
        public void onOutbountSessionMigration(org.codehaus.wadi.core.session.Session session) {
            notifyOutboundSessionMigration(session);
        }
        
    }

    protected Session newSession(HttpServletRequest request)
    {
        return new ClusteredSession(request);
    }

    public void complete(HttpSession session)
    {
        ClusteredSession clusteredSession = (ClusteredSession) session;
        clusteredSession.session.onEndAccess();
    }

    protected void addSession(Session session)
    {
        ClusteredSession clusteredSession = (ClusteredSession) session;
        synchronized (_idToSession)
        {
            _idToSession.put(clusteredSession.getClusterId(), clusteredSession);
        }
    }

    protected void removeSession(String idInCluster)
    {
        // Let MigrationListener handle session removal
    }

    public Session getSession(String idInCluster)
    {
        synchronized (_idToSession)
        {
            return _idToSession.get(idInCluster);
        }
    }

    public int getSessions()
    {
        synchronized (_idToSession)
        {
            return _idToSession.size();
        }
    }

    public Map getSessionMap()
    {
        throw new AssertionError("getSessionMap is never used.");
    }

    protected void invalidateSessions()
    {
    }

    private class MigrationListener implements SessionListener
    {

        public void notifyInboundSessionMigration(WadiSession session)
        {
            addSession(new ClusteredSession(session), false);
        }

        public void notifyOutboundSessionMigration(WadiSession session)
        {
            ClusteredSession clusteredSession = getClusteredSession(session);
            removeSession(clusteredSession, false);
        }

        public void notifySessionDestruction(WadiSession session)
        {
            ClusteredSession clusteredSession = getClusteredSession(session);
            removeSession(clusteredSession, true);
        }

        private ClusteredSession getClusteredSession(WadiSession session) throws AssertionError
        {
            ClusteredSession clusteredSession;
            synchronized (_idToSession)
            {
                clusteredSession = _idToSession.remove(session.getSessionId());
            }
            if (null == clusteredSession)
            {
                throw new AssertionError("Session [" + session + "] is undefined");
            }
            return clusteredSession;
        }

    }

    public class ClusteredSession extends Session
    {
        private final WadiSession session;

        protected ClusteredSession(HttpServletRequest request)
        {
            super(request);
            try
            {
                this.session = createSession(getClusterId());
            }
            catch (SessionAlreadyExistsException e)
            {
                throw (IllegalStateException) new IllegalStateException().initCause(e);
            }
            initValues();
        }

        protected ClusteredSession(WadiSession session)
        {
            super(System.currentTimeMillis(), session.getSessionId());
            this.session = session;
            initValues();
        }

        protected Map newAttributeMap()
        {
            return session.getState();
        }

        protected String getClusterId()
        {
            return super.getClusterId();
        }

        public void invalidate() throws IllegalStateException
        {
            super.doInvalidate();
            session.release();
        }
    }

}
