/*
 * @(#)ModeLoader.java 1.00 2005-10-12
 *
 * Copyright 2005 BeanSoft Studio. All rights reserved.
 * PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package studio.beansoft.syntax;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.text.Segment;

import org.gjt.sp.jedit.syntax.TokenHandler;
import org.gjt.sp.jedit.syntax.TokenMarker;
import org.gjt.sp.jedit.syntax.XModeHandler;

import com.microstar.xml.XmlException;
import com.microstar.xml.XmlParser;

/**
 * ModeLoader is used to load xml mode definition file. At first you must load
 * mode difination files from absolute file system or relative resource files.
 * Where the mode file be put must be readable using FileReader (that means, can't put 
 * in a jar file), eg: WEBAPP/WEB-INF/classes/mode/{catalog, java.xml, c.xml..}, or
 * on your application's file system.
 * Here is an sample code of usage:
 * <code>
 * ModeLoader.loadModeCatalog("/modes/catalog", true);
 * HtmlOutputTokenHandler tokenHandler = new HtmlOutputTokenHandler();
 * ModeLoader.parseTokens("public class Test() {}", "java", tokenHandler);
 * System.out.println(tokenHandler.getOutputText());
 * </code>
 * You can use any TokenHandler and any language defines in the catalog file.
 * 
 * Chinese documents:
 * XML ﷨ļļ. Ӿļ·ļ·
 * ģʽļ. Щļܹͨ FileReader ȡ(ζ
 * Щļŵ jar ļ),  WEBAPP/WEB-INF/classes/mode/{catalog, java.xml, c.xml..},
 * ĳļϵͳ.
 * Ǹʹ:
 * <code>
 * ModeLoader.loadModeCatalog("/modes/catalog", true);
 * HtmlOutputTokenHandler tokenHandler = new HtmlOutputTokenHandler();
 * ModeLoader.parseTokens("public class Test() {}", "java", tokenHandler);
 * System.out.println(tokenHandler.getOutputText());
 * </code>
 * ʹκ TokenHandler  catalog ļж.
 * 
 * @author BeanSoft
 * @version 1.00 2005-10-12
 */
public class ModeLoader {
	/**
	 * Stored modes.
	 */
	private static Vector modes = new Vector(50);

    private ModeLoader() {
    }
    
    /**
     * Parse tokens with an given luanguage type and given token handler.
     * @param string - code to parse
     * @param languageType - language string(see modes/catalog file)
     * @param tokenHandler - an TokenHandler object
     */
    public static void parseTokens(String string, String languageType, TokenHandler tokenHandler) {
		TokenMarker marker = getMode(languageType).getTokenMarker();

		StringTokenizer tokenizer = new StringTokenizer(string, "\r\n");// Parsing line by line
        
		TokenMarker.LineContext context = null;
	
		
		while(tokenizer.hasMoreElements()) {
			String line = tokenizer.nextElement().toString() + "\n";

			char[] lineChars = line.toCharArray();
			
			context = marker.markTokens(context,  tokenHandler,
			new Segment(
			lineChars, 0, lineChars.length));
		}
    }
    
	/**
	 * Parse tokens with an given luanguage type and given token handler.
	 * @param string - code to parse
	 * @param mode - an Mode object (mode name see modes/catalog file)
	 * @param tokenHandler - an TokenHandler object
	 */
	public static void parseTokens(String string, Mode mode, TokenHandler tokenHandler) {
		TokenMarker marker = mode.getTokenMarker();

		StringTokenizer tokenizer = new StringTokenizer(string, "\r\n");// Parsing line by line
        
		TokenMarker.LineContext context = null;
	
		
		while(tokenizer.hasMoreElements()) {
			String line = tokenizer.nextElement().toString() + "\n";

			char[] lineChars = line.toCharArray();
			
			context = marker.markTokens(context,  tokenHandler,
			new Segment(
			lineChars, 0, lineChars.length));
		}
	}    

   
	//{{{ loadMode() method
	/**
	 * Loads an XML-defined edit mode from the specified reader.
	 * This method is called by the Mode class.
	 * @param mode The edit mode
	 */
	/* package-private */ static void loadMode(Mode mode)
	{
		String oldfileName = mode.getProperty("file").toString();
		
		// Resolve the resource load problem on Windows at 2005.10.17 by BeanSoft
		if(oldfileName.startsWith("file:/")) {
			oldfileName = oldfileName.substring("file:/".length(), oldfileName.length());
		}
		
		final String fileName = oldfileName;

		final XmlParser parser = new XmlParser();
		XModeHandler xmh = new XModeHandler(mode.getName())
		{
			public void error(String what, Object subst)
			{
				int line = parser.getLineNumber();
				int column = parser.getColumnNumber();

				String msg = "";

				Object[] args = { fileName, new Integer(line),
					new Integer(column), msg };
				System.out.println("error" + "xmode-error" + args);
			}

			public TokenMarker getTokenMarker(String modeName)
			{
				Mode mode = getMode(modeName);
				if(mode == null)
					return null;
				else
					return mode.getTokenMarker();
			}
		};

		mode.setTokenMarker(xmh.getTokenMarker());

		Reader grammar = null;

		parser.setHandler(xmh);
		try
		{
			grammar = new BufferedReader(new FileReader(fileName));

			parser.parse(null, null, grammar);

			mode.setProperties(xmh.getModeProperties());
		}
		catch (Throwable e)
		{

			if (e instanceof XmlException)
			{
//				XmlException xe = (XmlException) e;
//				int line = xe.getLine();
//				String message = xe.getMessage();

				e.printStackTrace();
			}
		}
		finally
		{
			try
			{
				if(grammar != null)
					grammar.close();
			}
			catch(IOException io)
			{
				
			}
		}
	} //}}}
	
	//{{{ loadModeCatalog() method
	/**
	 * Loads a mode catalog file.
	 * @since jEdit 3.2pre2
	 */
	public static void loadModeCatalog(String path, boolean resource)
	{
		ModeCatalogHandler handler = new ModeCatalogHandler(
		getParentOfPath(path),resource);
		XmlParser parser = new XmlParser();
		parser.setHandler(handler);
		Reader in = null;
		try
		{
			InputStream _in;
			if(resource)
				_in = ModeLoader.class.getResourceAsStream(path);
			else
				_in = new FileInputStream(path);
			in = new BufferedReader(new InputStreamReader(_in));
			parser.parse(null, null, in);
		}
		catch(XmlException xe)
		{
//			int line = xe.getLine();
			String message = xe.getMessage();
			System.out.println(message);
		}
		catch(Exception e)
		{
			e.printStackTrace();
		}
		finally
		{
			try
			{
				if(in != null)
					in.close();
			}
			catch(IOException io)
			{

			}
		}
	} //}}}	
	
	//{{{ getMode() method
	/**
	 * Returns the edit mode with the specified name.
	 * @param name The edit mode
	 */
	public static Mode getMode(String name)
	{
		for(int i = 0; i < modes.size(); i++)
		{
			Mode mode = (Mode)modes.elementAt(i);
			if(mode.getName().equals(name))
				return mode;
		}
		return null;
	} //}}}
	
	//{{{ addMode() method
	/**
	 * Do not call this method. It is only public so that classes
	 * in the org.gjt.sp.jedit.syntax package can access it.
	 * @param mode The edit mode
	 */
	public static void addMode(Mode mode)
	{
		//Log.log(Log.DEBUG,jEdit.class,"Adding edit mode "
		//	+ mode.getName());

		modes.addElement(mode);
	} //}}}	
	
	//{{{ getParentOfPath() method
	/**
	 * Returns the parent of the specified path. This must be
	 * overridden to return a non-null value for browsing of this
	 * filesystem to work.
	 * @param path The path
	 * @since jEdit 2.6pre5
	 */
	public static String getParentOfPath(String path)
	{
		// ignore last character of path to properly handle
		// paths like /foo/bar/
		int lastIndex = path.length() - 1;
		while(lastIndex > 0
			&& (path.charAt(lastIndex) == File.separatorChar
			|| path.charAt(lastIndex) == '/'))
		{
			lastIndex--;
		}

		int count = Math.max(0,lastIndex);
		int index = path.lastIndexOf(File.separatorChar,count);
		if(index == -1)
			index = path.lastIndexOf('/',count);
		if(index == -1)
		{
			// this ensures that getFileParent("protocol:"), for
			// example, is "protocol:" and not "".
			index = path.lastIndexOf(':');
		}

		return path.substring(0,index + 1);
	} //}}}
	    
}
